#include "util.h"

// Convert Gamecube DTK/TRK/ADP tracks (essentially CD-XA ADPCM) to the
// nearest equivalent "DSP" ADPCM.

void transcode_frame(const char * framebuf, int channel, FILE * outfile) {
    int chanshift = (channel == 0 ? 0 : 4);
    uint8_t adp_frame_header = framebuf[0 + channel];
    int scale_log = 12 - (adp_frame_header & 0xf);
    int predictor = adp_frame_header >> 4;

    CHECK_ERROR(scale_log < 0 || scale_log > 16, "scale range");
    uint8_t dsp_frame_header = scale_log | (predictor << 4);

    uint8_t outframe[16];
    outframe[0] = outframe[8] = dsp_frame_header;

    for (int j = 0, k = 4; j < 2; j++) {
        for (int i = 0; i < 14; i += 2, k += 2) {
            uint8_t sample = (framebuf[k] >> chanshift) & 0xf;
            uint8_t b = sample << 4;

            sample = (framebuf[k+1] >> chanshift) & 0xf;
            b |= sample;

            outframe[j*8 + 1 + i/2] = b;
        }
    }

    put_bytes(outfile, outframe, 16);
}

int main(int argc, char ** argv) {
    for (int i = 1; i < argc; i++) {
        FILE * infile;
        FILE * outfile[2];
        const char * infile_name = argv[i];

        infile = fopen(infile_name, "rb");
        CHECK_ERRNO(!infile, "fopen");

        size_t inname_len = strlen(infile_name)+1;
        char infile_name_base[inname_len];
        char outnames[2][inname_len+5];

        strncpy(infile_name_base, infile_name, sizeof(infile_name_base));
        char * sep = strrchr(infile_name_base, '.');
        if (sep) {
            *sep = '\0';
        }

        strncpy(outnames[0], infile_name_base, sizeof(outnames[0]));
        strncat(outnames[0], "_L.dsp", sizeof(outnames[0]));
        outfile[0] = fopen(outnames[0], "w+b");
        CHECK_ERRNO(!outfile[0], "fopen");

        strncpy(outnames[1], infile_name_base, sizeof(outnames[1]));
        strncat(outnames[1], "_R.dsp", sizeof(outnames[1]));
        outfile[1] = fopen(outnames[1], "w+b");
        CHECK_ERRNO(!outfile[1], "fopen");

        // leave room for header
        CHECK_ERRNO(fseek(outfile[0], 0x60, SEEK_SET), "fseek");
        CHECK_ERRNO(fseek(outfile[1], 0x60, SEEK_SET), "fseek");

        char framebuf[32];
        int samples = 0;
        while (32 == fread(framebuf, 1, 32, infile)) {
            transcode_frame(framebuf, 0, outfile[0]);
            transcode_frame(framebuf, 1, outfile[1]);
            samples += 28;
        }

        uint8_t header[0x60];
        memset(header, 0, sizeof(header));

        // 0-3: sample count
        write_32_be(samples, header+0);

        // 4-7: nibble count
        write_32_be(samples/14*16, header+4);

        // 8-11: sample rate
        write_32_be(48000, header+8);

        // 12-13: loop flag (0)

        // 14-15: format (0: ADPCM)

        // 16-19: loop start, 20-23: loop end (0)

        // 24-27: ca ("current" nibble offset)
        write_32_be(2, header+24);

        // 28-59 filter coefficients
        {
            // these are the fixed filters used by XA
            const uint16_t coef[16] = {
                0,      0,
                0x3c,   0,
                0x73,  -0x34,
                0x62,  -0x37
            };

            for (int j = 0; j < 16; j++) {
                write_16_be(coef[j]<<5, header+28+j*2);
            }
        }

        // 60-61: gain (0)

        // 62-63: initial ps (needs to be set per file)
        //write_16_be(get_byte(0x60, outfile[]), header+62);

        // 64-65, 66-67: initial hist (0)
        // 68-69, 70-71, 72-73: loop ps, hist (0)

        // initial ps
        write_16_be(get_byte_seek(0x60, outfile[0]), header+62);
        // left channel header
        put_bytes_seek(0, outfile[0], header, 0x60);

        // initial ps
        write_16_be(get_byte_seek(0x60, outfile[1]), header+62);
        // right channel header
        put_bytes_seek(0, outfile[1], header, 0x60);


        fclose(infile);
        CHECK_ERRNO(EOF==fclose(outfile[0]), "fclose");
        CHECK_ERRNO(EOF==fclose(outfile[1]), "fclose");

        infile = outfile[0] = outfile[1] = 0;

        printf("%s -> %s, %s, %d samples\n", infile_name, outnames[0], outnames[1], samples);
    }

    return 0;
}
