#define __STDC_FORMAT_MACROS
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <stdlib.h>
#include <string.h>

/* fsbii 0.3 - convert multi-stream fsb into single-stream fsbs */

#define CHECK(x,msg) \
    do { \
        if (x) { \
            fprintf(stderr, "error " msg "\n"); \
            exit(EXIT_FAILURE); \
        } \
    } while (0)

#define CHECK_ERRNO(x,msg) \
    do { \
        if (x) { \
            perror("error " msg); \
            exit(EXIT_FAILURE); \
        } \
    } while (0)

#define CHECK_FILE(x,file,msg) \
    do { \
        if (x) { \
            if (feof(file)) { \
                fprintf(stderr, "error " msg ": eof\n"); \
            } \
            else \
            { \
                perror(msg); \
            } \
            exit(EXIT_FAILURE); \
        } \
    } while (0)


static inline int32_t read32bitLE(unsigned char *buf)
{
    int i;
    uint32_t value = 0;

    for ( i = 3; i >= 0; i-- )
    {
        value <<= 8;
        value |= buf[i];
    }

    return value;
}

static inline int16_t read16bitLE(unsigned char *buf)
{
    int i;
    uint32_t value = 0;

    for ( i = 1; i >= 0; i-- )
    {
        value <<= 8;
        value |= buf[i];
    }

    return value;
}

static inline void write32bitLE(int32_t value, unsigned char *buf)
{
    int i;
    uint32_t v = (uint32_t)value;

    for ( i = 0; i < 4; i++ )
    {
        buf[i] = v & 0xff;
        v >>= 8;
    }
}

void copy_bytes(FILE *infile, FILE *outfile, long offset, long count)
{
    int rc;
    size_t size_rc;
    unsigned char buf[0x800];

    rc = fseek(infile, offset, SEEK_SET);
    CHECK_ERRNO(rc == -1, "seek for copy");

    while (count > 0)
    {
        long dump_size = (count > sizeof(buf) ? sizeof(buf) : count);

        size_rc = fread(buf, 1, dump_size, infile);
        CHECK_FILE(size_rc != dump_size, infile, "reading for copy");

        size_rc = fwrite(buf, 1, dump_size, outfile);
        CHECK_FILE(size_rc != dump_size, outfile, "writing for copy");

        count -= dump_size;
    }
}

int main(int argc, char *argv[])
{
    FILE *infile;
    int rc;
    size_t size_rc;
    int32_t stream_count;
    int32_t table_size;
    int32_t body_size;
    long whole_file_size;
    int32_t header_size;
    unsigned char header[0x30];
    static const char fsb3headmagic[4]="FSB3"; /* not terminated */
    static const char fsb4headmagic[4]="FSB4"; /* not terminated */

    if (argc != 2)
    {
        printf("fsbii 0.3 - convert multi-stream fsb into single-stream fsbs\n"
                "usage: fsbii blah.fsb\n");
        exit(EXIT_FAILURE);
    }

    infile = fopen(argv[1],"rb");
    CHECK_ERRNO(infile == NULL, "opening input");

    /* get file size */
    {
        rc = fseek(infile, 0, SEEK_END);
        CHECK_ERRNO(rc == -1, "seeking to file end");

        whole_file_size = ftell(infile);
        CHECK_ERRNO(whole_file_size == -1, "getting file size");
    }

    /* read header */
    {
        rc = fseek(infile, 0, SEEK_SET);
        CHECK_ERRNO(rc == -1, "seeking to file start");

        size_rc = fread(header, 1, 4, infile);
        CHECK_FILE(size_rc != 4, infile, "reading magic");

        if (!memcmp(&header[0],fsb3headmagic,4))
        {
            printf("Type: FSB3\n");
            header_size = 0x18;
        }
        else if (!memcmp(&header[0],fsb4headmagic,4))
        {
            printf("Type: FSB4\n");
            header_size = 0x30;

            printf("Unfortunately not supported yet.\n");
            exit(EXIT_FAILURE);
        }
        else
        {
            CHECK(1, "unknown file type (not FSB3 or FSB4)\n");
        }

        /* read the rest of the hader */
        size_rc = fread(&header[4], 1, header_size-4, infile);
        CHECK_FILE(size_rc != header_size-4, infile, "reading header");

        stream_count = read32bitLE(&header[4]);
        CHECK(stream_count <= 0, "bad stream count");

        table_size = read32bitLE(&header[8]);
        CHECK(table_size <= 0, "bad table size");
        body_size = read32bitLE(&header[12]);
        CHECK(body_size <= 0, "bad body size");

        printf("Header: 0x%" PRIx32 " bytes\n", (uint32_t)header_size);
        printf("Table:  0x%" PRIx32 " bytes\n", (uint32_t)table_size);
        printf("Body:   0x%" PRIx32 " bytes\n", (uint32_t)body_size);
        printf("------------------\n");

        uint64_t total_size = (uint64_t)header_size +
                (uint64_t)table_size +
                (uint64_t)body_size;
        printf("Total:  0x%" PRIx64 " bytes\n", total_size);
        printf("File:   0x%lx bytes\n", whole_file_size);

        CHECK( whole_file_size < total_size ,
                "file size less than FSB3 size, truncated?");
        CHECK( whole_file_size - total_size > 0x800 ,
                "file size does not nearly match FSB3 size, concatenated?");

        if (stream_count == 1)
        {
            printf("Already a single stream.\n");
            exit(EXIT_SUCCESS);
        }

        printf("%" PRId32 " streams\n", stream_count);
    }

    /* copy each stream */
    {
        long table_offset = header_size;
        long body_offset = header_size + table_size;

        for (int i = 0; i < stream_count; i++) {
            int16_t entry_size;
            int32_t entry_file_size;
            static const char fsbext[] = ".fsb";
            const int entry_min_size = 0x28;
            unsigned char entry_buf[0x28];
            char name_buf[0x1e + sizeof(fsbext)]={0};
            FILE *outfile;

            rc = fseek(infile, table_offset, SEEK_SET);
            CHECK_ERRNO(rc, "seeking to table entry");

            size_rc = fread(entry_buf, 1, entry_min_size, infile);
            CHECK_FILE(size_rc != entry_min_size, infile,
                    "reading table entry header");

            entry_size = read16bitLE(&entry_buf[0]);
            CHECK(entry_size < entry_min_size, "entry too small");

            entry_file_size = read32bitLE(&entry_buf[0x24]);

            memcpy(name_buf, entry_buf+2, sizeof(name_buf));

            /* append .fsb to name */
            memcpy(name_buf+strlen(name_buf),fsbext,sizeof(fsbext));

            printf("%4d: %s"
                       " header 0x%02" PRIx32
                       " entry 0x%04" PRIx16
                       " body 0x%08" PRIx32
                       "\n",
                       i,
                       name_buf,
                       (uint32_t)header_size,
                       (uint16_t)entry_size,
                       (uint32_t)entry_file_size);

            /* open output */
            outfile = fopen(name_buf, "wb");
            CHECK_ERRNO(outfile == NULL, "opening output file");

            /* fill in the header */
            write32bitLE(1, &header[0x4]);
            write32bitLE(entry_size, &header[0x8]);
            write32bitLE(entry_file_size, &header[0xc]);

            /* write out the header */
            size_rc = fwrite(header, 1, header_size, outfile);
            CHECK_FILE(size_rc != header_size, outfile, "writing header");

            /* write out the table entry */
            copy_bytes(infile, outfile, table_offset, entry_size);

            /* write out the body */
            copy_bytes(infile, outfile, body_offset, entry_file_size);

            /* close */
            rc = fclose(outfile);
            CHECK_ERRNO(rc != 0, "closing output file");

            table_offset += entry_size;
            body_offset += entry_file_size;
        }
    }

    rc = fclose(infile);
    CHECK_ERRNO(rc != 0, "closing input file");

    return 0;
}
