/*
 * guessadx 0.1
 * by hcs
 *
 * find all possible encryption keys for an ADX file
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <limits.h>
#include <time.h>

int read16(unsigned char * buf) {
    return (buf[0]<<8)|buf[1];
}

int read32(unsigned char * buf) {
    return (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|buf[3];
}

void usage(const char * binname) {
    fprintf(stderr,"guessadx 0.1\n");
    fprintf(stderr,"usage: %s infile.adx\n",binname);
}

int score(int start, int mult, int add, unsigned short * scales, int scalecount) {
    int xor = start;
    int i;
    int total = 0;
    for (i=0;i<scalecount;i++) {
        if (scales[i] != 0)
            total += (scales[i] ^ xor)&0x7fff;
        xor = xor * mult + add;
    }
    return total;
}

int main(int argc, char ** argv) {
    FILE * infile = NULL;
    int bruteframe=0,bruteframecount=-1;
    unsigned char buf[18];
    int startoff, endoff;
    char * infilename = NULL;

    if (argc != 2) {
        usage(argv[0]);
        return 1;
    }

    /* parse command line */
    infilename = argv[1];

    /* open files */
    infile=fopen(infilename,"rb");
    if (!infile) {
        fprintf(stderr,"error opening %s\n",infilename);
        return 1;
    }

    /* read header */
    fseek(infile,0,SEEK_SET);
    fread(buf,16,1,infile);
    if (buf[0]!=0x80 || buf[1]!=0x00) {
        fprintf(stderr,"%s is not ADX\n",infilename);
        return 1;
    }
    if (buf[5]!=18) {
        fprintf(stderr,"%s does not have 18-byte frames, how odd... FAIL\n",infilename);
        return 1;
    }

    startoff=read16(buf+2)+4;
    endoff=(read32(buf+12)+31)/32*18*buf[7]+startoff;

    /* get version, encryption flag */
    fread(buf,4,1,infile);
    if (buf[3]!=8) {
        fprintf(stderr,"%s doesn't seem to be encrypted\n",infilename);
        return 1;
    }

    /* how many scales? */
    {
        int framecount=(endoff-startoff)/18;
        if (framecount<bruteframecount || bruteframecount<0)
            bruteframecount=framecount;
    }

    /* find longest run of nonzero frames */
    {
        int longest=-1,longest_length=-1;
        int i;
        int length=0;
        for (i=0;i<bruteframecount;i++) {
            static const unsigned char zeroes[18]={0};
            unsigned char buf[18];
            fseek(infile,startoff+i*18,SEEK_SET);
            fread(buf,18,1,infile);
            if (memcmp(zeroes,buf,18)) length++;
            else length=0;
            if (length > longest_length) {
                longest_length=length;
                longest=i-length+1;
                if (longest_length >= 0x8000) break;
            }
        }
        if (longest==-1) {
            fprintf(stderr,"no nonzero frames?\n");
            return 1;
        }
        bruteframecount = longest_length;
        bruteframe = longest;
    }

    {
        /* try to guess key */
#define MAX_FRAMES (INT_MAX/0x8000)
        int i,j,k,s;
        unsigned short * scales;
        unsigned short * prescales = NULL;
        int scales_to_do;
        static time_t starttime;
        int noback=1;

        /* allocate storage for scales */
        scales_to_do = (bruteframecount > MAX_FRAMES ? MAX_FRAMES : bruteframecount);
        scales = malloc(scales_to_do*sizeof(unsigned short));
        if (!scales) {
            fprintf(stderr,"error allocating memory for scales\n");
            return 0;
        }
        /* prescales are those scales before the first frame we test
         * against, we use these to compute the actual start */
        if (bruteframe > 0) {
            int i;
            /* allocate memory for the prescales */
            prescales = malloc(bruteframe*sizeof(unsigned short));
            if (!prescales) {
                fprintf(stderr,"error allocating memory for prescales\n");
                return 0;
            }
            /* read the prescales */
            for (i=0; i<bruteframe; i++) {
                unsigned char buf[2];
                fseek(infile,startoff+i*18,SEEK_SET);
                if (fread(buf,2,1,infile)!=1) {
                    fprintf(stderr,"read error\n");
                    return 1;
                }
                prescales[i] = read16(buf);
            }
        }

        /* read in the scales */
        {
            int i;
            for (i=0; i <= scales_to_do; i++) {
                unsigned char buf[2];
                fseek(infile,startoff+(bruteframe+i)*18,SEEK_SET);
                if (fread(buf,2,1,infile)!=1) {
                    fprintf(stderr,"read error\n");
                    return 1;
                }
                scales[i] = read16(buf);
            }
        }

        fprintf(stderr,"\n");
        starttime = time(NULL);
        /* do it! */

        /* guess possible low bits for start */
        for (i=0;i<=0x1fff;i++) {
            int start = i+(scales[0]&0x6000);
            /* status report */
            if (i>=1) {
                char messagebuf[100];
                time_t etime = time(NULL)-starttime;
                time_t donetime = 0x2000*etime/i-etime;

                sprintf(messagebuf,"%4x %3d%% %8ld minute%c elapsed %8ld minute%c left (maybe)",
                        i,
                        i*100/0x2000,
                        (etime/60),
                        (etime/60)!=1 ? 's' : ' ',
                        (donetime/60),
                        (donetime/60)!=1 ? 's' : ' ');
                if (!noback) {
                    int i;
                    for (i=0;i<strlen(messagebuf);i++)
                        fprintf(stderr,"\b");
                }
                fprintf(stderr,"%s",messagebuf);
                fflush(stderr);
                noback=0;
            }

            /* guess multiplier */
            /* it is assumed that only odd multipliers are used */
            for (j=1;j<=0x7fff;j+=2) {
                int mult = j;

                /* guess low bits for second scale */
                for (k=0;k<=0x1fff;k++) {
                    int second = (scales[1]&0x6000) + k;
                    int add = (second - (start*mult))&0x7fff;
                    int xor = second * mult + add;

                    /* test */
                    for (s=2;s<scales_to_do &&
                            (scales[s]&0x6000)==(xor&0x6000);s++) {
                        xor = xor * mult + add;
                    }

                    /* if we tested all values, we have a match */
                    if (s==scales_to_do) {
                        fprintf(stderr,"\n");
                        fflush(stderr);
                        noback=1;

                        /* if our "start" isn't actually on the first frame,
                         * find possible real start values */
                        if (bruteframe>0) {
                            int realstart;
                            int i;
                            for (realstart = 0; realstart < 0x7fff; realstart++) {
                                int xor = realstart;
                                for (i=0;i<bruteframe &&
                                        ((prescales[i]&0x6000)==(xor&0x6000) ||
                                         prescales[i]==0);
                                        i++) {
                                    xor = xor * mult + add;
                                }

                                if (i==bruteframe && (xor&0x7fff)==start) {
                                    printf("-s %4x -m %4x -a %4x (error %d)\n",realstart,mult,add,score(start,mult,add,scales,scales_to_do)+score(realstart,mult,add,prescales,bruteframe));
                                }
                            }
                        } else {
                            printf("-s %4x -m %4x -a %4x (error %d)\n",start,mult,add,score(start,mult,add,scales,scales_to_do));
                        }
                        fflush(stdout);
                    }
                } /* end add for loop */
            } /* end mult for loop */
        } /* end start for loop */
    } /* end key guess section */
    fprintf(stderr,"\n");
    return 0;
}
